const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const crypto = require('crypto');
const { google } = require('googleapis');
const logger = require('./logger');

const PLAYLIST_PAGE_SIZE = 50;
const VIDEO_PAGE_SIZE = 50;
const DEFAULT_LIBRARY_CACHE_MS = 60 * 1000;

function parseISODuration(isoDuration) {
  if (!isoDuration || typeof isoDuration !== 'string') {
    return 0;
  }
  const match = isoDuration.match(
    /P(?:([0-9]+)Y)?(?:([0-9]+)M)?(?:([0-9]+)W)?(?:([0-9]+)D)?(?:T(?:([0-9]+)H)?(?:([0-9]+)M)?(?:([0-9]+)S)?)?/
  );
  if (!match) {
    return 0;
  }
  const [
    ,
    years,
    months,
    weeks,
    days,
    hours,
    minutes,
    seconds
  ] = match.map(value => (value ? Number(value) : 0));

  const totalSeconds =
    (((years * 365 + months * 30 + weeks * 7 + days) * 24 + hours) * 60 + minutes) * 60 + seconds;

  return Number.isFinite(totalSeconds) ? totalSeconds : 0;
}

function determineVideoType(durationSeconds, liveBroadcastContent, title = '', description = '') {
  if (liveBroadcastContent && liveBroadcastContent !== 'none') {
    return 'live';
  }
  
  // Check if video has #Shorts hashtag in title or description
  const hasShortsHashtag = /#shorts/i.test(title + ' ' + description);
  
  // Only classify as Shorts if:
  // 1. Duration is <= 60 seconds (YouTube's Shorts limit)
  // 2. AND has #Shorts hashtag in title or description
  // This prevents misclassifying regular short videos as Shorts
  if (Number.isFinite(durationSeconds) && durationSeconds > 0 && durationSeconds <= 60 && hasShortsHashtag) {
    return 'shorts';
  }
  
  return 'videos';
}

class YouTubeService {
  constructor({ app, googleOAuth, getMainWindow }) {
    this.app = app;
    this.googleOAuth = googleOAuth;
    this.getMainWindow = getMainWindow;

    this.stores = new Map(); // accountId -> { videos: {}, loaded: boolean }
    this.storePaths = new Map(); // accountId -> storePath

    this.activeJobs = new Map(); // jobId -> job
    this.jobsByVideoPath = new Map(); // compositeKey -> jobId
    this.accountCaches = new Map(); // accountId -> { channel, library }
    this.getAllRecordsPromises = new Map(); // accountId -> Promise to prevent duplicate calls
  }

  getAccountCache(accountId) {
    const key = accountId || '__active__';
    if (!this.accountCaches.has(key)) {
      this.accountCaches.set(key, {
        channel: {
          fetchedAt: 0,
          channelId: null,
          uploadsPlaylistId: null,
          channelTitle: null,
          channelThumbnailUrl: null,
          channelCustomUrl: null
        },
        library: {
          fetchedAt: 0,
          snapshot: null
        }
      });
    }
    return this.accountCaches.get(key);
  }

  resetAccountCache(accountId, scope = 'all') {
    const key = accountId || '__active__';
    if (!this.accountCaches.has(key)) {
      return;
    }
    const cache = this.accountCaches.get(key);
    if (scope === 'all' || scope === 'channel') {
      cache.channel = {
        fetchedAt: 0,
        channelId: null,
        uploadsPlaylistId: null,
        channelTitle: null,
        channelThumbnailUrl: null,
        channelCustomUrl: null
      };
    }
    if (scope === 'all' || scope === 'library') {
      cache.library = {
        fetchedAt: 0,
        snapshot: null
      };
    }
  }

  getJobMapKey(videoPath, accountId) {
    return `${videoPath}::${accountId || 'default'}`;
  }

  getAbsoluteVideoPath(videoPath) {
    return path.resolve(videoPath);
  }

  getStorePath(accountId) {
    if (!accountId) {
      throw new Error('accountId is required to get store path');
    }
    
    if (!this.storePaths.has(accountId)) {
      const userDataDir = this.app.getPath('userData');
      // Sanitize accountId for filename
      const safeAccountId = accountId.replace(/[^a-zA-Z0-9_-]/g, '_');
      const storePath = path.join(userDataDir, `youtube-videos-${safeAccountId}.json`);
      this.storePaths.set(accountId, storePath);
    }
    return this.storePaths.get(accountId);
  }
  
  getStore(accountId) {
    if (!this.stores.has(accountId)) {
      this.stores.set(accountId, { videos: {}, loaded: false });
    }
    return this.stores.get(accountId);
  }

  getCachePath(accountId) {
    if (!accountId) {
      return null;
    }
    const userDataDir = this.app.getPath('userData');
    const cacheDir = path.join(userDataDir, 'youtube-cache');
    // Sanitize accountId for filename
    const safeAccountId = accountId.replace(/[^a-zA-Z0-9_-]/g, '_');
    return path.join(cacheDir, `${safeAccountId}.json`);
  }

  async loadLibraryCache(accountId) {
    if (!accountId) {
      return null;
    }
    const cachePath = this.getCachePath(accountId);
    if (!cachePath) {
      return null;
    }
    try {
      const content = await fsp.readFile(cachePath, 'utf8');
      const parsed = JSON.parse(content);
      if (parsed && typeof parsed === 'object') {
        // Validate cache structure - be lenient with accountId match (cache might have been created with different accountId format)
        // As long as we have videos array, use the cache
        if (Array.isArray(parsed.videos)) {
          // Ensure accountId is set correctly
          if (!parsed.accountId || parsed.accountId !== accountId) {
            parsed.accountId = accountId;
          }
          return parsed;
        }
      }
    } catch (error) {
      // Cache file doesn't exist or is invalid - return null
      if (error.code !== 'ENOENT' && process.env.NODE_ENV !== 'test') {
        logger.warn('Failed to load YouTube library cache:', error.message);
      }
    }
    return null;
  }

  async saveLibraryCache(accountId, libraryData) {
    if (!accountId || !libraryData) {
      return false;
    }
    const cachePath = this.getCachePath(accountId);
    if (!cachePath) {
      return false;
    }
    try {
      const directory = path.dirname(cachePath);
      await fsp.mkdir(directory, { recursive: true });
      
      // Ensure accountId is set in the data
      const dataToSave = {
        ...libraryData,
        accountId
      };
      
      await fsp.writeFile(cachePath, JSON.stringify(dataToSave, null, 2), 'utf8');
      return true;
    } catch (error) {
      if (process.env.NODE_ENV !== 'test') {
        logger.warn('Failed to save YouTube library cache:', error.message);
      }
      return false;
    }
  }

  async addVideoToCache(accountId, videoData) {
    if (!accountId || !videoData || !videoData.videoId) {
      return false;
    }
    
    // Load existing cache
    const existingCache = await this.loadLibraryCache(accountId);
    if (!existingCache) {
      // No cache exists, can't add to it
      return false;
    }
    
    // Ensure videos array exists
    if (!Array.isArray(existingCache.videos)) {
      existingCache.videos = [];
    }
    
    // Check if video already exists
    const existingIndex = existingCache.videos.findIndex(v => v && v.videoId === videoData.videoId);
    
    // Normalize video data to match library format
    const normalizedVideo = {
      videoId: String(videoData.videoId),
      title: videoData.title || 'Untitled Video',
      description: videoData.description || '',
      thumbnails: videoData.thumbnails || {},
      thumbnailUrl: videoData.thumbnailUrl || null,
      publishedAt: videoData.publishedAt || new Date().toISOString(),
      channelTitle: videoData.channelTitle || existingCache.channelTitle || '',
      channelId: videoData.channelId || existingCache.channelId || null,
      durationSeconds: videoData.durationSeconds || 0,
      liveBroadcastContent: videoData.liveBroadcastContent || 'none',
      privacy: videoData.privacy || videoData.privacyStatus || 'unlisted',
      privacyStatus: videoData.privacyStatus || videoData.privacy || 'unlisted',
      embeddable: videoData.embeddable !== undefined ? videoData.embeddable : true,
      status: 'uploaded',
      type: videoData.type || determineVideoType(
        videoData.durationSeconds || 0, 
        videoData.liveBroadcastContent || 'none',
        videoData.title || '',
        videoData.description || ''
      ),
      url: videoData.url || (videoData.videoId ? `https://youtu.be/${videoData.videoId}` : null),
      playlistIds: Array.isArray(videoData.playlistIds) ? videoData.playlistIds : [],
      accountId: accountId
    };
    
    if (existingIndex >= 0) {
      // Update existing video
      existingCache.videos[existingIndex] = normalizedVideo;
    } else {
      // Add new video at the beginning (most recent first)
      existingCache.videos.unshift(normalizedVideo);
    }
    
    // Update fetchedAt timestamp
    existingCache.fetchedAt = new Date().toISOString();
    
    // Save updated cache
    return await this.saveLibraryCache(accountId, existingCache);
  }

  async ensureStoreLoaded(accountId) {
    if (!accountId) {
      throw new Error('accountId is required to load store');
    }
    
    const store = this.getStore(accountId);
    if (store.loaded) {
      return;
    }

    const storePath = this.getStorePath(accountId);
    try {
      const content = await fsp.readFile(storePath, 'utf8');
      const parsed = JSON.parse(content);
      if (parsed && typeof parsed === 'object') {
        if (parsed.videos && typeof parsed.videos === 'object') {
          store.videos = parsed.videos;
        } else {
          store.videos = {};
        }
      }
    } catch (error) {
      if (error.code !== 'ENOENT') {
        // Only warn if it's not a "file not found" error
        logger.warn('Failed to load YouTube store for account:', accountId, error.message);
      }
      store.videos = {};
    }

    store.loaded = true;
  }

  async preloadStore(accountId) {
    if (!accountId) {
      return;
    }
    const store = this.getStore(accountId);
    if (store.loaded) {
      return;
    }
    // Silently preload the store (await to ensure it completes)
    try {
      await this.ensureStoreLoaded(accountId);
    } catch {
      // Ignore errors during preload - will be handled when actually needed
    }
  }

  async saveStore(accountId) {
    if (!accountId) {
      throw new Error('accountId is required to save store');
    }
    
    await this.ensureStoreLoaded(accountId);
    const store = this.getStore(accountId);
    const storePath = this.getStorePath(accountId);
    const directory = path.dirname(storePath);
    await fsp.mkdir(directory, { recursive: true });
    const dataToSave = { videos: store.videos };
    logger.log('[YouTube] Saving store to:', storePath, 'with', Object.keys(store.videos).length, 'videos');
    await fsp.writeFile(storePath, JSON.stringify(dataToSave, null, 2), 'utf8');
    logger.log('[YouTube] Store file written successfully');
  }

  async getRecord(videoPath, accountId) {
    if (!accountId) {
      throw new Error('accountId is required to get record');
    }
    
    await this.ensureStoreLoaded(accountId);
    const store = this.getStore(accountId);
    const resolved = this.getAbsoluteVideoPath(videoPath);
    return store.videos[resolved] || null;
  }

  async setRecord(videoPath, record, accountId) {
    if (!accountId) {
      throw new Error('accountId is required when setting a video record');
    }
    
    logger.log('[YouTube] setRecord called:', { videoPath, accountId, hasVideoId: !!record.videoId });
    
    await this.ensureStoreLoaded(accountId);
    const store = this.getStore(accountId);
    const resolved = this.getAbsoluteVideoPath(videoPath);
    const existing = store.videos[resolved] || {};
    
    const payload = {
      ...existing,
      ...record,
      accountId,
      updatedAt: new Date().toISOString()
    };
    
    store.videos[resolved] = payload;
    logger.log('[YouTube] About to save store for account:', accountId, 'Store path:', this.getStorePath(accountId));
    await this.saveStore(accountId);
    logger.log('[YouTube] Store saved successfully for account:', accountId);
    return payload;
  }

  async removeRecord(videoPath, accountId) {
    if (!accountId) {
      throw new Error('accountId is required to remove record');
    }
    
    await this.ensureStoreLoaded(accountId);
    const store = this.getStore(accountId);
    const resolved = this.getAbsoluteVideoPath(videoPath);
    
    if (store.videos[resolved]) {
      delete store.videos[resolved];
      await this.saveStore(accountId);
    }
  }

  async getAllRecords(accountId) {
    if (!accountId) {
      throw new Error('accountId is required to get all records');
    }
    
    // Check if store is already loaded - if so, return immediately without any logging or async work
    const store = this.getStore(accountId);
    if (store.loaded) {
      return { ...store.videos };
    }
    
    // Check if there's already an in-flight request for this account
    const existingPromise = this.getAllRecordsPromises.get(accountId);
    if (existingPromise) {
      return await existingPromise;
    }
    
    // Create a new promise for this request and set it immediately (atomic check-and-set)
    // Note: This loads from local disk cache, not YouTube API
    const promise = (async () => {
      try {
        await this.ensureStoreLoaded(accountId);
        const loadedStore = this.getStore(accountId);
        return { ...loadedStore.videos };
      } finally {
        // Remove the promise from cache once it completes
        this.getAllRecordsPromises.delete(accountId);
      }
    })();
    
    // Cache the promise immediately (before awaiting) to prevent race conditions
    this.getAllRecordsPromises.set(accountId, promise);
    
    return await promise;
  }

  getActiveJobByVideoPath(videoPath, accountId = null) {
    const jobKey = this.getJobMapKey(videoPath, accountId);
    const jobId = this.jobsByVideoPath.get(jobKey);
    if (!jobId) {
      return null;
    }
    return this.activeJobs.get(jobId) || null;
  }

  async getAuthYouTubeClient(options = {}) {
    const authClient = await this.googleOAuth.getAuthenticatedClient(undefined, options);
    let accountId = authClient.youtubeAccountId || (options && options.accountId) || null;
    if (!accountId) {
      try {
        const status = await this.googleOAuth.getAuthStatus();
        if (status && status.activeAccountId) {
          accountId = status.activeAccountId;
        }
      } catch (error) {
        // ignore status retrieval errors here; downstream will handle missing account id.
      }
    }
    const youtube = google.youtube({
      version: 'v3',
      auth: authClient
    });
    return { youtube, accountId: accountId || null };
  }

  getWebContentsCandidates(preferredWebContents) {
    const targets = [];
    if (preferredWebContents && !preferredWebContents.isDestroyed()) {
      targets.push(preferredWebContents);
    }
    const mainWindow = this.getMainWindow ? this.getMainWindow() : null;
    const mainContents = mainWindow && mainWindow.webContents ? mainWindow.webContents : null;
    if (mainContents && !mainContents.isDestroyed() && (!preferredWebContents || mainContents.id !== preferredWebContents.id)) {
      targets.push(mainContents);
    }
    return targets;
  }

  sendToRenderers(channel, payload, preferredWebContents) {
    const targets = this.getWebContentsCandidates(preferredWebContents);
    for (const target of targets) {
      try {
        target.send(channel, payload);
      } catch (error) {
        // ignore failures to send
      }
    }
  }

  emitUploadProgress(job, preferredWebContents, extra = {}) {
    const payload = {
      jobId: job.id,
      videoPath: job.videoPath,
      accountId: job.accountId || null,
      progress: job.progress,
      status: job.status,
      ...extra
    };
    this.sendToRenderers('youtube.upload.progress', payload, preferredWebContents);
  }

  emitUploadCompleted(job, preferredWebContents, payload) {
    this.sendToRenderers('youtube.upload.completed', {
      jobId: job.id,
      videoPath: job.videoPath,
      accountId: job.accountId || null,
      ...payload
    }, preferredWebContents);
  }

  async getChannelIdentifiers(youtube, accountId = null) {
    const cacheMaxAge = 5 * 60 * 1000;
    const now = Date.now();
    const cache = this.getAccountCache(accountId).channel;
    if (
      cache.uploadsPlaylistId
      && now - cache.fetchedAt < cacheMaxAge
    ) {
      logger.log('[YouTube API] getChannelIdentifiers: Using cached channel info for account:', accountId);
      return {
        uploadsPlaylistId: cache.uploadsPlaylistId,
        channelId: cache.channelId,
        channelTitle: cache.channelTitle,
        accountId
      };
    }

    logger.log('[YouTube API] CALLING youtube.channels.list for account:', accountId);
    const response = await youtube.channels.list({
      mine: true,
      part: ['contentDetails', 'snippet']
    });
    logger.log('[YouTube API] youtube.channels.list completed for account:', accountId);

    const items = response && response.data && Array.isArray(response.data.items)
      ? response.data.items
      : [];

    if (items.length === 0) {
      throw new Error('Unable to load YouTube channel information.');
    }

    const channel = items[0];
    const uploadsPlaylistId = channel
      && channel.contentDetails
      && channel.contentDetails.relatedPlaylists
      ? channel.contentDetails.relatedPlaylists.uploads
      : null;

    if (!uploadsPlaylistId) {
      throw new Error('Uploads playlist not found for this channel.');
    }

    const channelId = channel.id || null;
    const snippet = channel && channel.snippet ? channel.snippet : {};
    const thumbnails = snippet.thumbnails || {};
    const channelTitle = snippet.title || null;
    const channelThumbnailUrl = thumbnails.high?.url || thumbnails.medium?.url || thumbnails.default?.url || null;
    const channelCustomUrl = snippet.customUrl || null;

    cache.fetchedAt = now;
    cache.uploadsPlaylistId = uploadsPlaylistId;
    cache.channelId = channelId;
    cache.channelTitle = channelTitle;
    cache.channelThumbnailUrl = channelThumbnailUrl;
    cache.channelCustomUrl = channelCustomUrl;

    if (accountId) {
      this.googleOAuth.updateAccountProfile(accountId, {
        channelId,
        channelTitle,
        channelCustomUrl,
        channelThumbnailUrl
      }).catch(() => {});
    }

    return { uploadsPlaylistId, channelId, channelTitle, accountId };
  }

  async fetchChannelVideosFromUploadsPlaylist(youtube, uploadsPlaylistId, maxVideos = 200) {
    if (!uploadsPlaylistId) {
      return [];
    }
    // Note: This uses the uploads playlist internally to discover videos
    // The uploads playlist is a system playlist automatically created by YouTube
    // We use it only for video discovery, not for playlist functionality
    logger.log('[YouTube API] CALLING youtube.playlistItems.list for uploads playlist (internal use only), playlistId:', uploadsPlaylistId, 'maxVideos:', maxVideos);
    const videoIds = new Set(); // Use Set to automatically deduplicate
    let nextPageToken = undefined;
    let pageCount = 0;
    const PLAYLIST_PAGE_SIZE = 50;
    
    while (videoIds.size < maxVideos) {
      const remaining = maxVideos - videoIds.size;
      if (remaining <= 0) {
        break;
      }
      pageCount++;
      logger.log('[YouTube API] youtube.playlistItems.list page', pageCount, 'for uploads playlist (internal use only), playlistId:', uploadsPlaylistId);
      const response = await youtube.playlistItems.list({
        playlistId: uploadsPlaylistId,
        part: ['contentDetails'],
        maxResults: Math.min(PLAYLIST_PAGE_SIZE, remaining),
        pageToken: nextPageToken
      });
      const pageItems = response && response.data && Array.isArray(response.data.items)
        ? response.data.items
        : [];
      
      let addedInThisPage = 0;
      for (const item of pageItems) {
        if (item && item.contentDetails && item.contentDetails.videoId) {
          if (videoIds.size >= maxVideos) {
            break; // Stop adding if we've reached the limit
          }
          videoIds.add(item.contentDetails.videoId);
          addedInThisPage++;
        }
      }
      
      // If we didn't get any new items or we've reached the limit, stop
      if (addedInThisPage === 0 || videoIds.size >= maxVideos) {
        break;
      }
      
      nextPageToken = response && response.data ? response.data.nextPageToken : undefined;
      if (!nextPageToken) {
        break; // No more pages available
      }
    }
    
    const videoIdsArray = Array.from(videoIds);
    logger.log('[YouTube API] youtube.playlistItems.list completed for uploads playlist (internal use only), total video IDs:', videoIdsArray.length);
    return videoIdsArray;
  }

  async fetchVideosByIds(youtube, videoIds = []) {
    const results = [];
    if (!Array.isArray(videoIds) || videoIds.length === 0) {
      logger.log('[YouTube API] fetchVideosByIds: No video IDs provided, returning empty');
      return results;
    }
    logger.log('[YouTube API] CALLING youtube.videos.list for', videoIds.length, 'video IDs');
    for (let i = 0; i < videoIds.length; i += VIDEO_PAGE_SIZE) {
      const batch = videoIds.slice(i, i + VIDEO_PAGE_SIZE);
      logger.log('[YouTube API] youtube.videos.list batch', Math.floor(i / VIDEO_PAGE_SIZE) + 1, 'of', Math.ceil(videoIds.length / VIDEO_PAGE_SIZE), '(', batch.length, 'videos)');
      const response = await youtube.videos.list({
        id: batch,
        part: ['snippet', 'contentDetails', 'status']
      });
      const items = response && response.data && Array.isArray(response.data.items)
        ? response.data.items
        : [];
      for (const item of items) {
        const snippet = item && item.snippet ? item.snippet : {};
        const contentDetails = item && item.contentDetails ? item.contentDetails : {};
        const status = item && item.status ? item.status : {};
        const durationSeconds = parseISODuration(contentDetails.duration);
        const liveBroadcastContent = snippet.liveBroadcastContent || 'none';
        const title = snippet.title || 'Untitled Video';
        const description = snippet.description || '';
        const normalized = {
          videoId: item.id || null,
          title,
          description,
          thumbnails: snippet.thumbnails || {},
          thumbnailUrl: snippet.thumbnails
            ? (snippet.thumbnails.high && snippet.thumbnails.high.url)
              || (snippet.thumbnails.medium && snippet.thumbnails.medium.url)
              || (snippet.thumbnails.default && snippet.thumbnails.default.url)
            : null,
          publishedAt: snippet.publishedAt || null,
          channelTitle: snippet.channelTitle || '',
          channelId: snippet.channelId || null,
          durationSeconds,
          liveBroadcastContent,
          privacy: status.privacyStatus || null,
          embeddable: status.embeddable !== undefined ? status.embeddable : true,
          status: 'uploaded',
          type: determineVideoType(durationSeconds, liveBroadcastContent, title, description),
          url: item.id ? `https://youtu.be/${item.id}` : null,
          playlistIds: []
        };
        results.push(normalized);
      }
    }
    logger.log('[YouTube API] youtube.videos.list completed, fetched', results.length, 'videos out of', videoIds.length, 'requested');
    return results;
  }

  async fetchChannelPlaylists(youtube, channelId, maxPlaylists = 50) {
    const playlists = [];
    let nextPageToken = undefined;
    do {
      const remaining = maxPlaylists - playlists.length;
      if (remaining <= 0) {
        break;
      }
      const response = await youtube.playlists.list({
        mine: true,
        part: ['snippet', 'contentDetails', 'status'],
        maxResults: Math.min(PLAYLIST_PAGE_SIZE, remaining),
        pageToken: nextPageToken
      });
      const pageItems = response && response.data && Array.isArray(response.data.items)
        ? response.data.items
        : [];
      for (const item of pageItems) {
        const snippet = item && item.snippet ? item.snippet : {};
        const contentDetails = item && item.contentDetails ? item.contentDetails : {};
        const status = item && item.status ? item.status : {};
        const lowerTitle = (snippet.title || '').toLowerCase();
        const lowerDescription = (snippet.description || '').toLowerCase();
        const isPodcast = lowerTitle.includes('podcast') || lowerDescription.includes('podcast');
        playlists.push({
          id: item.id || null,
          title: snippet.title || 'Untitled Playlist',
          description: snippet.description || '',
          thumbnails: snippet.thumbnails || {},
          itemCount: contentDetails.itemCount || 0,
          channelTitle: snippet.channelTitle || '',
          channelId: snippet.channelId || channelId || null,
          publishedAt: snippet.publishedAt || null,
          updatedAt: snippet.publishedAt || null,
          privacy: status.privacyStatus || null,
          type: isPodcast ? 'podcast' : 'playlist'
        });
      }
      nextPageToken = response && response.data ? response.data.nextPageToken : undefined;
    } while (nextPageToken);
    return playlists.filter(playlist => playlist.id);
  }

  async getLibrarySnapshot(options = {}) {
    const cacheMs = typeof options.cacheMs === 'number' ? options.cacheMs : DEFAULT_LIBRARY_CACHE_MS;
    const now = Date.now();
    const force = options.force === true;

    // Only fetch from YouTube API if:
    // 1. force is true (manual refresh button clicked), OR
    // 2. Local cache is empty/missing
    // This prevents unnecessary API calls on app start when cache exists

    // First check in-memory cache
    const accountId = options.accountId || null;
    const accountCache = this.getAccountCache(accountId);
    const libraryCache = accountCache.library;

    if (
      !force
      && libraryCache.snapshot
      && now - libraryCache.fetchedAt < cacheMs
    ) {
      logger.log('[YouTube API] getLibrarySnapshot: Using in-memory cache for account:', accountId, '(age:', Math.round((now - libraryCache.fetchedAt) / 1000), 'seconds)');
      return libraryCache.snapshot;
    }

    // If not forcing, check disk cache - use it regardless of age to avoid API calls on startup
    if (!force) {
      logger.log('[YouTube API] getLibrarySnapshot: Checking disk cache for account:', accountId);
      const diskCache = await this.loadLibraryCache(accountId);
      if (diskCache && diskCache.videos && Array.isArray(diskCache.videos)) {
        // Load disk cache into memory and return it (even if old - we don't want to call API on startup)
        const cacheAge = diskCache.fetchedAt ? Math.round((now - new Date(diskCache.fetchedAt).getTime()) / 1000) : 'unknown';
        logger.log('[YouTube API] getLibrarySnapshot: Using disk cache for account:', accountId, 'with', diskCache.videos.length, 'videos (cache age:', cacheAge, 'seconds)');
        const snapshot = {
          accountId: diskCache.accountId || accountId,
          channelId: diskCache.channelId || null,
          channelTitle: diskCache.channelTitle || null,
          fetchedAt: diskCache.fetchedAt || new Date().toISOString(),
          videos: diskCache.videos || [],
          playlists: diskCache.playlists || [],
          playlistItems: diskCache.playlistItems || {}
        };
        
        // Update in-memory cache
        const fetchedAtMs = snapshot.fetchedAt ? new Date(snapshot.fetchedAt).getTime() : now;
        libraryCache.fetchedAt = fetchedAtMs;
        libraryCache.snapshot = snapshot;
        
        return snapshot;
      } else {
        logger.log('[YouTube API] getLibrarySnapshot: No disk cache found for account:', accountId);
      }
    }

    // Only make API calls if force is true
    // If we got here without force, cache should have been found above
    // As a safety measure, if force is false and we have no cache, return empty instead of calling API
    if (!force) {
      // This should never happen if cache loading worked correctly
      logger.warn('[YouTube API] getLibrarySnapshot: No cache found and force=false, returning empty snapshot to avoid API call for account:', accountId);
      // Return empty snapshot instead of calling API to avoid quota usage
      return {
        accountId: accountId || null,
        channelId: null,
        channelTitle: null,
        fetchedAt: new Date().toISOString(),
        videos: [],
        playlists: [],
        playlistItems: {}
      };
    }
    
    // Force refresh requested - make API calls
    logger.log('[YouTube API] getLibrarySnapshot: FORCE REFRESH requested - will call YouTube API for account:', accountId);
    const { youtube, accountId: resolvedAccountId } = await this.getAuthYouTubeClient({ accountId });
    const finalAccountId = resolvedAccountId || accountId;

    logger.log('[YouTube API] getLibrarySnapshot: Starting API calls for account:', finalAccountId);
    const { uploadsPlaylistId, channelId, channelTitle } = await this.getChannelIdentifiers(youtube, finalAccountId);
    logger.log('[YouTube API] getLibrarySnapshot: Got channel identifiers, channelId:', channelId);

    const maxVideos = Number.isFinite(options.maxVideos) ? options.maxVideos : 200;

    // Fetch video IDs from uploads playlist (system playlist, used internally only for video discovery)
    // Note: This is the most reliable way to get all videos from a channel
    // We use the uploads playlist internally but don't expose playlist functionality to users
    const videoIds = await this.fetchChannelVideosFromUploadsPlaylist(youtube, uploadsPlaylistId, maxVideos);
    const uniqueVideoIds = videoIds.filter(Boolean); // Just filter out any falsy values

    const videos = await this.fetchVideosByIds(youtube, uniqueVideoIds);
    for (const video of videos) {
      video.playlistIds = Array.isArray(video.playlistIds) ? video.playlistIds : [];
      video.accountId = finalAccountId;
      if (!video.channelTitle && channelTitle) {
        video.channelTitle = channelTitle;
      }
    }

    const snapshot = {
      accountId: finalAccountId,
      channelId,
      channelTitle,
      fetchedAt: new Date().toISOString(),
      videos,
      playlists: [],
      playlistItems: {}
    };

    libraryCache.fetchedAt = now;
    libraryCache.snapshot = snapshot;

    // Save cache to disk
    await this.saveLibraryCache(finalAccountId, snapshot).catch(() => {
      // Ignore save errors - cache will still work in memory
    });

    logger.log('[YouTube API] getLibrarySnapshot: API calls completed for account:', finalAccountId, 'fetched', snapshot.videos.length, 'videos');
    return snapshot;
  }

  async startUpload({ videoPath, metadata = {}, webContents, accountId: requestedAccountId = null }) {
    const resolvedPath = this.getAbsoluteVideoPath(videoPath);

    if (!fs.existsSync(resolvedPath)) {
      throw new Error('Source video file not found.');
    }

    const fileStat = await fsp.stat(resolvedPath);
    if (!fileStat.isFile()) {
      throw new Error('The selected path is not a file.');
    }

    const { youtube, accountId: resolvedAccountIdRaw } = await this.getAuthYouTubeClient({ accountId: requestedAccountId });
    const resolvedAccountId = resolvedAccountIdRaw || requestedAccountId || null;

    logger.log('[YouTube] startUpload - accountId resolution:', {
      requestedAccountId,
      resolvedAccountIdRaw,
      resolvedAccountId
    });

    if (!resolvedAccountId) {
      throw new Error('YouTube account ID is required to upload. Please ensure you are signed in to a YouTube account.');
    }

    if (this.getActiveJobByVideoPath(resolvedPath, resolvedAccountId)) {
      throw new Error('A YouTube upload is already in progress for this video and account.');
    }

    const jobId = crypto.randomUUID();
    const job = {
      id: jobId,
      videoPath: resolvedPath,
      accountId: resolvedAccountId,
      status: 'uploading',
      progress: 0,
      startedAt: Date.now()
    };

    const jobKey = this.getJobMapKey(resolvedPath, resolvedAccountId);

    this.activeJobs.set(jobId, job);
    this.jobsByVideoPath.set(jobKey, jobId);

    const defaultTitle = path.basename(resolvedPath, path.extname(resolvedPath));
    const title = (metadata.title || defaultTitle).slice(0, 100);
    const description = metadata.description || '';
    const privacy = (metadata.privacy || 'unlisted').toLowerCase();

    const categoryId = metadata.categoryId || '22';

    const snippet = {
      title,
      description,
      categoryId
    };

    const status = {
      privacyStatus: ['public', 'private', 'unlisted'].includes(privacy) ? privacy : 'unlisted',
      selfDeclaredMadeForKids: false
    };

    const fileSize = fileStat.size;

    const runUpload = async () => {
      const fileStream = fs.createReadStream(resolvedPath);
      let uploadedBytes = 0;
      let lastProgressEmit = -1;
      let cleanupCalled = false;

      const cleanupJob = () => {
        this.resetAccountCache(resolvedAccountId, 'library');
        if (cleanupCalled) {
          return;
        }
        cleanupCalled = true;
        this.activeJobs.delete(jobId);
        this.jobsByVideoPath.delete(jobKey);
        if (!fileStream.destroyed) {
          fileStream.destroy();
        }
      };

      fileStream.on('data', (chunk) => {
        uploadedBytes += chunk.length;
        if (fileSize > 0) {
          const progress = Math.min(100, (uploadedBytes / fileSize) * 100);
          if (progress - lastProgressEmit >= 1 || progress === 100) {
            lastProgressEmit = progress;
            job.progress = progress;
            job.status = 'uploading';
            this.emitUploadProgress(job, webContents);
          }
        }
      });

      try {
        const response = await youtube.videos.insert({
          part: ['snippet', 'status'],
          requestBody: {
            snippet,
            status
          },
          media: {
            body: fileStream
          }
        });

        job.status = 'completed';
        job.progress = 100;

        const data = response && response.data ? response.data : {};
        const uploadedVideo = {
          videoId: data.id || null,
          title: data.snippet && data.snippet.title ? data.snippet.title : title,
          description: data.snippet && data.snippet.description ? data.snippet.description : description,
          privacy: data.status && data.status.privacyStatus ? data.status.privacyStatus : status.privacyStatus,
          categoryId: data.snippet && data.snippet.categoryId ? data.snippet.categoryId : (snippet.categoryId || null),
          publishedAt: data.snippet && data.snippet.publishedAt ? data.snippet.publishedAt : new Date().toISOString(),
          url: data.id ? `https://youtu.be/${data.id}` : null,
          accountId: resolvedAccountId
        };

        if (uploadedVideo.videoId) {
          try {
            logger.log('[YouTube] Saving upload record:', {
              videoPath: resolvedPath,
              videoId: uploadedVideo.videoId,
              accountId: resolvedAccountId
            });
            const savedRecord = await this.setRecord(resolvedPath, uploadedVideo, resolvedAccountId);
            logger.log('[YouTube] Record saved successfully:', savedRecord);
            this.sendToRenderers('youtube.video.updated', {
              videoPath: resolvedPath,
              video: uploadedVideo,
              accountId: resolvedAccountId
            }, webContents);
          } catch (error) {
            logger.error('[YouTube] Failed to save upload record:', error);
            logger.error('[YouTube] Error details:', {
              videoPath: resolvedPath,
              accountId: resolvedAccountId,
              error: error.message,
              stack: error.stack
            });
            // Still send the update event even if saving failed
            this.sendToRenderers('youtube.video.updated', {
              videoPath: resolvedPath,
              video: uploadedVideo,
              accountId: resolvedAccountId
            }, webContents);
          }
        }

        this.emitUploadCompleted(job, webContents, {
          success: true,
          uploadedVideo
        });
      } catch (error) {
        job.status = 'failed';
        const message = error && error.message ? error.message : 'YouTube upload failed.';
        this.emitUploadCompleted(job, webContents, {
          success: false,
          error: message,
          progress: job.progress || 0
        });
        throw error;
      } finally {
        cleanupJob();
      }
    };

    this.emitUploadProgress(job, webContents);

    runUpload().catch((error) => {
      const message = error && error.message ? error.message : error;
      if (process.env.NODE_ENV !== 'test') {
        logger.warn('YouTube upload failed:', message);
      }
    });

    return {
      id: job.id,
      videoPath: job.videoPath,
      accountId: job.accountId,
      status: job.status,
      progress: job.progress
    };
  }

  async updateMetadata({ videoPath, videoId, metadata = {}, webContents }) {
    // Handle cases where videoPath is not provided (e.g., editing from YouTube library)
    const resolvedPath = videoPath ? this.getAbsoluteVideoPath(videoPath) : null;
    
    // Try to get accountId from metadata first, then from auth client
    const { youtube, accountId: accountIdFromClient } = await this.getAuthYouTubeClient({ accountId: metadata.accountId });
    let resolvedAccountId = accountIdFromClient || metadata.accountId || null;
    
    // If we have a videoPath, try to get the record for the resolved account
    let existingRecord = null;
    if (resolvedPath && resolvedAccountId) {
      try {
        existingRecord = await this.getRecord(resolvedPath, resolvedAccountId);
      } catch (error) {
        // Record doesn't exist for this account, that's okay
        existingRecord = null;
      }
    }
    
    const targetVideoId = videoId || (existingRecord && existingRecord.videoId);

    if (!targetVideoId) {
      throw new Error('No existing YouTube video mapping found. videoId is required when videoPath is not provided.');
    }

    // Ensure we have a resolved accountId
    if (!resolvedAccountId) {
      // Try to get from auth status as last resort
      try {
        const status = await this.googleOAuth.getAuthStatus();
        if (status && status.activeAccountId) {
          resolvedAccountId = status.activeAccountId;
        }
      } catch (error) {
        // Ignore errors
      }
    }

    let existingSnippet = null;
    let existingStatus = null;

    try {
      const current = await youtube.videos.list({
        id: [targetVideoId],
        part: ['snippet', 'status']
      });
      if (current && current.data && Array.isArray(current.data.items) && current.data.items.length > 0) {
        const video = current.data.items[0];
        existingSnippet = video && video.snippet ? video.snippet : null;
        existingStatus = video && video.status ? video.status : null;
      }
    } catch (fetchError) {
      if (process.env.NODE_ENV !== 'test') {
        logger.warn('Failed to load existing YouTube video details:', fetchError && fetchError.message ? fetchError.message : fetchError);
      }
    }

    // Get title from metadata, existing record, or YouTube API response
    const title = metadata.title 
      || (existingRecord && existingRecord.title) 
      || (existingSnippet && existingSnippet.title)
      || (resolvedPath ? path.basename(resolvedPath, path.extname(resolvedPath)) : 'YouTube Video');
    const description = metadata.description != null 
      ? metadata.description 
      : (existingRecord && existingRecord.description) 
      || (existingSnippet && existingSnippet.description) 
      || '';
    const privacy = (metadata.privacy 
      || (existingRecord && existingRecord.privacy) 
      || (existingStatus && existingStatus.privacyStatus)
      || 'unlisted').toLowerCase();

    const categoryId = metadata.categoryId
      || (existingSnippet && existingSnippet.categoryId)
      || (existingRecord && existingRecord.categoryId)
      || '22';

    const snippetPayload = {
      title: title.slice(0, 100),
      description,
      categoryId
    };

    if (metadata.tags && Array.isArray(metadata.tags) && metadata.tags.length > 0) {
      snippetPayload.tags = metadata.tags;
    } else if (existingSnippet && Array.isArray(existingSnippet.tags) && existingSnippet.tags.length > 0) {
      snippetPayload.tags = existingSnippet.tags;
    }

    if (metadata.defaultLanguage) {
      snippetPayload.defaultLanguage = metadata.defaultLanguage;
    } else if (existingSnippet && existingSnippet.defaultLanguage) {
      snippetPayload.defaultLanguage = existingSnippet.defaultLanguage;
    }

    if (metadata.defaultAudioLanguage) {
      snippetPayload.defaultAudioLanguage = metadata.defaultAudioLanguage;
    } else if (existingSnippet && existingSnippet.defaultAudioLanguage) {
      snippetPayload.defaultAudioLanguage = existingSnippet.defaultAudioLanguage;
    }

    const statusPayload = {
      privacyStatus: ['public', 'private', 'unlisted'].includes(privacy) ? privacy : 'unlisted'
    };

    // CRITICAL: Preserve embeddable setting - don't let it default to false
    if (metadata.embeddable != null) {
      statusPayload.embeddable = Boolean(metadata.embeddable);
    } else if (existingStatus && existingStatus.embeddable != null) {
      statusPayload.embeddable = existingStatus.embeddable;
    } else {
      // Default to true if not specified (YouTube's default is true)
      statusPayload.embeddable = true;
    }

    if (metadata.selfDeclaredMadeForKids != null) {
      statusPayload.selfDeclaredMadeForKids = Boolean(metadata.selfDeclaredMadeForKids);
    } else if (existingStatus && existingStatus.selfDeclaredMadeForKids != null) {
      statusPayload.selfDeclaredMadeForKids = existingStatus.selfDeclaredMadeForKids;
    }

    try {
      const response = await youtube.videos.update({
        part: ['snippet', 'status'],
        requestBody: {
          id: targetVideoId,
          snippet: snippetPayload,
          status: statusPayload
        }
      });

      const data = response && response.data ? response.data : {};
      const updatedRecord = {
        videoId: targetVideoId,
        title: data.snippet && data.snippet.title ? data.snippet.title : title,
        description: data.snippet && data.snippet.description ? data.snippet.description : description,
        privacy: data.status && data.status.privacyStatus ? data.status.privacyStatus : privacy,
        embeddable: data.status && data.status.embeddable !== undefined ? data.status.embeddable : (statusPayload.embeddable !== undefined ? statusPayload.embeddable : true),
        categoryId: data.snippet && data.snippet.categoryId ? data.snippet.categoryId : categoryId,
        publishedAt: data.snippet && data.snippet.publishedAt ? data.snippet.publishedAt : (existingRecord && existingRecord.publishedAt) || new Date().toISOString(),
        url: existingRecord && existingRecord.url ? existingRecord.url : `https://youtu.be/${targetVideoId}`,
        accountId: resolvedAccountId
      };

      // Only save record if we have a videoPath
      if (resolvedPath && resolvedAccountId) {
        await this.setRecord(resolvedPath, updatedRecord, resolvedAccountId);
      }

      this.sendToRenderers('youtube.video.updated', {
        videoPath: resolvedPath || null,
        video: updatedRecord,
        accountId: resolvedAccountId
      }, webContents);

      // Don't reset the cache for library videos without local path
      // The library will be refreshed naturally, and we want to keep the video accessible
      if (resolvedPath) {
        this.resetAccountCache(resolvedAccountId, 'library');
      }

      return updatedRecord;
    } catch (error) {
      throw new Error(error && error.message ? error.message : 'Failed to update YouTube metadata.');
    }
  }

  async deleteVideo({ videoPath, videoId, accountId: requestedAccountId, webContents }) {
    // videoPath is optional - YouTube library videos may only have videoId
    const resolvedPath = videoPath ? this.getAbsoluteVideoPath(videoPath) : null;
    
    // Try to get accountId from parameter first, then from auth client
    const { youtube, accountId: accountIdFromClient } = await this.getAuthYouTubeClient({ accountId: requestedAccountId });
    let resolvedAccountId = requestedAccountId || accountIdFromClient || null;
    
    // Get record for the resolved account (only if we have a videoPath)
    let existingRecord = null;
    if (resolvedPath && resolvedAccountId) {
      try {
        existingRecord = await this.getRecord(resolvedPath, resolvedAccountId);
      } catch (error) {
        // Record doesn't exist for this account, that's okay
        existingRecord = null;
      }
    }
    
    const targetVideoId = videoId || (existingRecord && existingRecord.videoId);

    if (!targetVideoId) {
      throw new Error('No YouTube video mapping to remove.');
    }

    // Ensure we have a resolved accountId
    if (!resolvedAccountId) {
      // Try to get from auth status as last resort
      try {
        const status = await this.googleOAuth.getAuthStatus();
        if (status && status.activeAccountId) {
          resolvedAccountId = status.activeAccountId;
        }
      } catch (error) {
        // Ignore errors
      }
    }

    if (!resolvedAccountId) {
      throw new Error('YouTube account ID is required to delete video.');
    }

    try {
      await youtube.videos.delete({ id: targetVideoId });

      // Only remove record if we have a videoPath
      if (resolvedPath) {
        await this.removeRecord(resolvedPath, resolvedAccountId);
      }

      this.sendToRenderers('youtube.video.removed', {
        videoPath: resolvedPath,
        videoId: targetVideoId,
        accountId: resolvedAccountId
      }, webContents);

      if (resolvedAccountId) {
        this.resetAccountCache(resolvedAccountId, 'library');
      }

      return true;
    } catch (error) {
      throw new Error(error && error.message ? error.message : 'Failed to remove YouTube video.');
    }
  }

  async listAccounts() {
    const status = await this.googleOAuth.getAuthStatus();
    return status || { authenticated: false, accounts: [], activeAccountId: null };
  }

  async selectAccount(accountId) {
    await this.googleOAuth.setActiveAccount(accountId);
    this.resetAccountCache(accountId, 'all');
    this.accountCaches.delete('__active__');
    const status = await this.googleOAuth.getAuthStatus();
    return status;
  }

  async removeAccount(accountId) {
    await this.googleOAuth.removeAccount(accountId);
    const cacheKey = accountId || '__active__';
    this.accountCaches.delete(cacheKey);
    this.accountCaches.delete('__active__');
    const status = await this.googleOAuth.getAuthStatus();
    return status;
  }
}

module.exports = YouTubeService;

